#!/bin/sh
###############################################################################
#
#                     Update systems root filesystem
#

# application tag for logging
APP_TAG=SystemUpdater

# list of files to backup
BACKUP_FILE_LIST=$UPD_ROOT/backup.list
# where to put backed up filesystem
BACKUP_DIR=$BACKUP_DIR
# location of backed up root filesystem
BACKUP_FS_TAR=$BACKUP_DIR/rootfs.tar


	
##
#	Helper functions
#

log()
{
#   $1 - Log messge
    echo -e "$(date) - $1" | tee -a $UPD_LOG
    #logger -t $APP_TAG $1
}
error()
{
    log "$2"
    echo -e "$(date) - $1" | tee -a $ERR_LOG
    exit $1
}



##
#	Update from the initial rootfs layout
#
UpdateFromVersion1()
{
    log "Starting update from version 1"

    ##
    #	Update filesystem
    #
    log "Update filesystem"
    mkdir -p /rootfs/oldfs

    gzip -cd $UPD_ROOT/rootfs.tar.gz | tar -x -C /rootfs || log "Failed to update filesystem!"

    cp -dpf $UPD_ROOT/mountall.sh /etc/init.d/
    chmod u+rwx /etc/init.d/mountall.sh

    cp /root/gw5Setup.bin /rootfs/root/
}

##
#	Update from the buildroot rootfs layout
#
UpdateFromVersion2()
{
    log "Starting update from version 2"
    

    kernel_updated=0
    product_file=/usr/local/etc/telem/product
    NANDWRITE=/usr/sbin/nandwrite
    uname_m=$(uname -m)
    cpu_hardware=$(grep "Hardware" /proc/cpuinfo)

    FLASH_ERASE=/usr/sbin/flash_erase
    FLASH_ERASEALL=/usr/sbin/flash_eraseall
    
    kernel_image=$UPD_ROOT/zImage

    log "$cpu_hardware"

    cpu_type_in_file=$(cat $UPD_ROOT/Board)
    
    test -e $product_file && {
        product_name_sys=$(cat $product_file)
        product_name_upd=$(cat $UPD_ROOT/product)
        if [ _"$product_name_sys" != _"$product_name_upd" ]; then
            error 1 "Product name mismatch ('$product_name_sys' != '$product_name_upd'), stopping update!"
        fi
    }

    echo $cpu_hardware | grep $cpu_type_in_file || {
        error 1 "CPU type mismatch ('$cpu_hardware' != '$cpu_type_in_file'), stopping update."
    }

    echo $cpu_hardware | grep PXA270 > /dev/null && {
#       Voipac PXA270 Module

        KERNEL_MTD=/dev/mtd1
        
        if [ -f $FLASH_ERASE ]; then
            erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
        else
            FLASH_ERASE=/usr/sbin/flash_eraseall
            erase_cmd="$FLASH_ERASE $KERNEL_MTD"
        fi
    }

    echo $cpu_hardware | grep VMX25 > /dev/null && {
#       Voipac Technologies VMX25 Board

        KERNEL_MTD=/dev/mtd2
        erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
    }

    echo $cpu_hardware | grep VMX51 > /dev/null && {
#       Voipac Technologies VMX51 Board

        KERNEL_MTD=/dev/mtd3
        erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
    }

    echo $cpu_hardware | grep VMX53 > /dev/null && {
#       Voipac Technologies VMX53 Board

        KERNEL_MTD=/dev/mtd2
        erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
    }

    echo $cpu_hardware | grep M502-SOM > /dev/null && {
#       Artila Electronics M502-SOM Board

        kernel_image=$UPD_ROOT/uImage
        KERNEL_MTD=/dev/mtd3
        erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
    }

    test -e $kernel_image && {
        log "Starting kernel update"

        # Check for necessary tools
        if [ -e $FLASH_ERASE ]; then

            if [ -e $NANDWRITE ]; then
                # erase flash
                # log "Run: '$erase_cmd'"
                eval $erase_cmd
                if [ "$?" = "0" ]; then

                    log "Old kernel erased."

                    log "Writing kernel '$kernel_image' to flash '$KERNEL_MTD'"

                    # write new kernel to flash
                    nandwrite -p $KERNEL_MTD $kernel_image
                    if [ "$?" = "0" ]; then
                        log "New kernel written to flash."

                        kernel_updated=1
    
                        # remove old kernel modules
                        rm -rf /lib/modules/*
                    else
                        error 1 "Failed to write new kernel to flash."
                    fi
                else
                    error 1 "Failed to erase flash."
                fi

            else
                error 1 "Utility '$NANDWRITE' not available"
            fi
        else
            error 1 "Utility '$FLASH_ERASE' not available"
        fi
    }
    
    
    ##
    #	Update filesystem
    #
    log "Update filesystem"
    #mkdir -p /rootfs/oldfs

    _current=/etc/current

    mkdir -p $_current
    chmod u=rw,o= $_current

    echo "Backing up groups and users"
    cp -pf /etc/group $_current/
    cp -pf /etc/passwd $_current/
    cp -pf /etc/shadow $_current/
    
    echo "Removing old WWW directory"
    ! rm -rf /var/www/*
    
    ! rm -rf /usr/local/bin/post_update/*

    rootfs_tar=$UPD_ROOT/rootfs.tar
    rootfs_tgz=$rootfs_tar.gz
    rootfs_tbz2=$rootfs_tar.bz2
    rootfs_txz=$rootfs_tar.xz
    rootfs_tlz=$rootfs_tar.lzma

    if [ -e $rootfs_tgz ]; then
        unpack_cmd="gzip -cd $rootfs_tgz | tar -x -C /"
    elif [ -e $rootfs_tlz ]; then
        unpack_cmd="lzcat $rootfs_tlz | tar -x -C /"
    elif [ -e $rootfs_tbz2 ]; then
        unpack_cmd="bzcat $rootfs_tbz2 | tar -x -C /"
    elif [ -e $rootfs_txz ]; then
        unpack_cmd="xzcat $rootfs_txz | tar -x -C /"
    elif [ -e $rootfs_tar ]; then
        unpack_cmd="tar -x -C / $rootfs_tar"
    fi

    eval $unpack_cmd || error 1 "Failed to update filesystem!"

    test -e $_current && {
        echo "Restoring groups and users"
        mv $_current/* /etc/
        rm -rf $_current
    }
    
    chmod +x /usr/local/bin/telem/check_permissions

    /usr/local/bin/telem/check_permissions /

    /usr/local/bin/telem/post_update.sh
}


##
#			The Script
#

##
#	Clean unneccessary files/directories
#
rm -R /root/upd/backup
rm -R /oldfs/root/upd
# remove old logs
rm -R /oldfs/root/log
# remove gzipped logs
rm /root/log/archive/*.gz


[ -e /usr/bin/tar ] && rm /bin/tar
##
# Select update to perform
#


if [ -e /linuxrc ]; then
	# gateway has update from buildroot
	UpdateFromVersion2
else
	UpdateFromVersion1
fi


exit 0